//  2011 IDesign Inc.
// Pytania? Komentarze? Odwied
// http://www.idesign.net

using System.Collections.Generic;
using System.Diagnostics;
using System.Threading;
using System.Transactions;

namespace ServiceModelEx
{
   /// <summary>
   /// Chroni menedera zasobw, zapewniajc izolacj transakcji (tylko na poziomie implementacji Serializable)
   /// </summary>
   public class TransactionalLock
   {
      // Transakcja, ktra prbuje uzyska blokad zajt przez inn transakcj, jest umieszczana w kolejce
      LinkedList<KeyValuePair<Transaction,ManualResetEvent>> m_PendingTransactions = new LinkedList<KeyValuePair<Transaction,ManualResetEvent>>();
      Transaction m_OwningTransaction;

      Transaction OwningTransaction
      {
         get 
         {
            lock(this)
            {
               return m_OwningTransaction;
            }
         }
         set 
         {
            lock(this)
            {
               m_OwningTransaction = value;
            }
         }
      }
      public bool Locked
      {
         get
         {
            return OwningTransaction != null;
         }
      }      
      /// <summary>
      /// Uzyskuje blokad na wyczno. Jeli inna transakcja dysponuje t blokad, bieca transakcja jest blokowana i umieszczana w kolejce. Jeli bieca transakcja dysponuje ju blokad, metoda Lock() nie podejmuje adnych dziaa. 
      /// </summary>
      public void Lock()
      {
         Lock(Transaction.Current);
      }
      void Lock(Transaction transaction)
      {
         bool taken = false;

         Monitor.Enter(this,ref taken);

         Debug.Assert(taken);

         if(OwningTransaction == null)
         {
            if(transaction == null)
            {
               Monitor.Exit(this); 
               return;
            }
            else
            {
               Debug.Assert(transaction.IsolationLevel == IsolationLevel.Serializable);

               // Uzyskuje blokad
               OwningTransaction = transaction;
               Monitor.Exit(this); 
               return;
            }
         }
         else // Inna transakcja zajmuje blokad
         {
            // Sprawdza, czy jest to bieca transakcja
            if(OwningTransaction == transaction)
            {
               Monitor.Exit(this); 
               return;
            }
            else // Potrzebuje blokady
            {
               ManualResetEvent manualEvent = new ManualResetEvent(false);

               KeyValuePair<Transaction,ManualResetEvent> pair;
               pair = new KeyValuePair<Transaction,ManualResetEvent>(transaction,manualEvent);
               m_PendingTransactions.AddLast(pair);

               if(transaction != null)
               {
                  Debug.Assert(transaction.TransactionInformation.Status == TransactionStatus.Active);
                  // Ponieaw transakcja moe zosta przerwana lub przekroczy limit czasu w czasie blokowania zasobu, naley usun blokad i usun transakcj z kolejki po zakoczeniu dziaania
                  transaction.TransactionCompleted += delegate
                                                      {
                                                         lock(this)
                                                         {
                                                            // Dana para klucz-warto moga ju zosta usunita z kolejki transakcji
                                                            m_PendingTransactions.Remove(pair);
                                                         }
                                                         lock(manualEvent)// To rozwizanie pozwala unikn sytuacji wycigu wskutek zamknicia uchwytu pomidzy testem a ustawieniem
                                                         {
                                                            if(manualEvent.SafeWaitHandle.IsClosed == false)
                                                            {
                                                               manualEvent.Set();
                                                            }
                                                         }
                                                      };
               }
               Monitor.Exit(this); 
               // Blokuje transakcj lub wtek wywoujcy
               manualEvent.WaitOne();
               lock(manualEvent) // To rozwizania pozwala unikn sytuacji wycigu w zwizku z ustawieniem uchwytu przez pozostae wtki
               {
                  manualEvent.Close();
               }
            }
         }
      }
      // Zwalnia blokad transakcji i umoliwia uzyskanie blokady przez nastpn transakcj w kolejce. 
      public void Unlock()
      {
         Debug.Assert(Locked);
         lock(this)
         {
            OwningTransaction = null;
                  
            LinkedListNode<KeyValuePair<Transaction,ManualResetEvent>> node = null;

            if(m_PendingTransactions.Count > 0)
            {
               node = m_PendingTransactions.First;
               m_PendingTransactions.RemoveFirst();
            }
            if(node != null)
            {
               Transaction transaction = node.Value.Key;
               ManualResetEvent manualEvent = node.Value.Value;
               Lock(transaction);
               lock(manualEvent) // To rozwizanie pozwala unikn sytuacji wycigu wskutek zamknicia uchwytu pomidzy testem a ustawieniem
               {
                  if(manualEvent.SafeWaitHandle.IsClosed == false)
                  {
                     manualEvent.Set();
                  }
               }
            }
         }
      }
   }
}
